/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.hdfs;

import com.google.common.annotations.VisibleForTesting;
import com.oracle.bmc.hdfs.store.BmcDataStore;
import com.oracle.bmc.hdfs.store.BmcDataStoreFactory;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.ParentNotDirectoryException;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.util.Progressable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class BmcFilesystemImpl
extends FileSystem {
    private static final Logger LOG = LoggerFactory.getLogger(BmcFilesystemImpl.class);
    private static final PathLengthComparator PATH_LENGTH_COMPARATOR = new PathLengthComparator();
    private Path workingDirectory;
    private BmcDataStore dataStore;
    private URI uri;
    private volatile boolean isInitialized;

    BmcFilesystemImpl() {
    }

    public void initialize(URI uri, Configuration configuration) throws IOException {
        if (this.isInitialized) {
            return;
        }
        LOG.info("Attempting to initialize filesystem with URI {}", (Object)uri);
        UriParser uriParser = new UriParser(uri);
        String scheme = uriParser.getScheme();
        if (scheme.equals("oraclebmc")) {
            LOG.warn("Using deprecated scheme {}", (Object)uri.getScheme());
        }
        super.initialize(uri, configuration);
        super.setConf(configuration);
        String namespace = uriParser.getNamespace();
        if (namespace == null) {
            throw new IllegalArgumentException("Namespace cannot be empty");
        }
        String bucket = uriParser.getBucket();
        if (bucket == null) {
            throw new IllegalArgumentException("Bucket cannot be empty");
        }
        LOG.info("Initialized filesystem for namespace {} and bucket {}", (Object)namespace, (Object)bucket);
        this.uri = URI.create(scheme + "://" + uriParser.getAuthority());
        this.dataStore = new BmcDataStoreFactory(configuration).createDataStore(namespace, bucket, this.statistics);
        String username = System.getProperty("user.name");
        this.workingDirectory = super.makeQualified(new Path("/user", username));
        this.isInitialized = true;
        LOG.info("Setting working directory to {}, and initialized uri to {}", (Object)this.workingDirectory, (Object)this.uri);
    }

    public String getScheme() {
        return "oci";
    }

    public FSDataOutputStream append(Path path, int bufferSize, Progressable progress) throws IOException {
        throw new UnsupportedOperationException("Appending is not supported with BMC Object Store");
    }

    public FSDataOutputStream create(Path path, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(path, permission, overwrite, bufferSize, replication, blockSize, progress, true);
    }

    protected FSDataOutputStream create(Path path, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress, boolean isRecursive) throws IOException {
        LOG.debug("Attempting to create path {}, overwrite {}, bufferSize {}", new Object[]{path, overwrite, bufferSize});
        FileStatus existingFile = this.getNullableFileStatus(path);
        if (existingFile != null) {
            if (existingFile.isDirectory()) {
                throw new FileAlreadyExistsException("Cannot create file, path already exists as a directory: " + path);
            }
            if (!overwrite) {
                throw new FileAlreadyExistsException("Path already exists, and no overwrite allowed: " + path);
            }
            LOG.debug("Found existing file at path, deleting");
            this.dataStore.delete(path);
        } else if (isRecursive) {
            LOG.debug("No existing file at path {}, verifying all directories exist with mkdirs", (Object)path);
            this.mkdirs(path.getParent(), permission);
        } else if (this.getNullableFileStatus(path.getParent()) == null) {
            throw new FileNotFoundException("Cannot create file " + path + ", the parent directory does not exist");
        }
        return new FSDataOutputStream(this.dataStore.openWriteStream(path, bufferSize, progress), this.statistics);
    }

    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.create(f, permission, flags.contains(CreateFlag.OVERWRITE), bufferSize, replication, blockSize, progress, false);
    }

    public boolean delete(Path path, boolean recursive) throws IOException {
        LOG.debug("Requested to delete {}, recursive {}", (Object)path, (Object)recursive);
        FileStatus status = this.getNullableFileStatus(path);
        if (status == null) {
            LOG.debug("No file at path {} found, nothing to delete", (Object)path);
            return false;
        }
        if (status.isFile()) {
            LOG.info("Deleting file");
            this.dataStore.delete(path);
            return true;
        }
        boolean isEmptyDirectory = this.dataStore.isEmptyDirectory(path);
        if (isEmptyDirectory) {
            if (status.getPath().isRoot()) {
                LOG.info("Empty root directory, nothing to delete");
                return true;
            }
            LOG.info("Deleting empty directory");
            this.dataStore.deleteDirectory(path);
            return true;
        }
        if (!recursive) {
            throw new IOException("Attempting to delete a directory that is not empty, and recursive delete not specified: " + path);
        }
        ArrayList<FileStatus> directories = new ArrayList<FileStatus>();
        directories.add(status);
        ArrayList<Path> directoriesToDelete = new ArrayList<Path>();
        LOG.debug("Recursively deleting directory");
        while (!directories.isEmpty()) {
            FileStatus directory = (FileStatus)directories.remove(0);
            Path directoryPath = this.ensureAbsolutePath(directory.getPath());
            List<FileStatus> entries = this.dataStore.listDirectory(directoryPath);
            for (FileStatus entry : entries) {
                if (entry.isDirectory()) {
                    directories.add(entry);
                    continue;
                }
                this.dataStore.delete(this.ensureAbsolutePath(entry.getPath()));
            }
            directoriesToDelete.add(directoryPath);
        }
        Collections.sort(directoriesToDelete, PATH_LENGTH_COMPARATOR);
        for (Path directoryToDelete : directoriesToDelete) {
            this.dataStore.deleteDirectory(directoryToDelete);
        }
        return true;
    }

    public FileStatus getFileStatus(Path path) throws IOException {
        LOG.debug("Requested file status for {}", (Object)path);
        Path absolutePath = this.ensureAbsolutePath(path);
        FileStatus fileStatus = this.dataStore.getFileStatus(absolutePath);
        if (fileStatus == null) {
            throw new FileNotFoundException("No file found at path: " + path);
        }
        return fileStatus;
    }

    private FileStatus getNullableFileStatus(Path path) throws IOException {
        try {
            return this.getFileStatus(path);
        }
        catch (FileNotFoundException e) {
            return null;
        }
    }

    public FileStatus[] listStatus(Path path) throws FileNotFoundException, IOException {
        LOG.debug("Requested listStatus for {}", (Object)path);
        FileStatus status = this.getFileStatus(path);
        if (status.isFile()) {
            return new FileStatus[]{status};
        }
        return this.dataStore.listDirectory(path).toArray(new FileStatus[0]);
    }

    public boolean mkdirs(Path path, FsPermission permission) throws IOException {
        LOG.debug("Requested mkdirs on path {}", (Object)path);
        Path currentPath = path;
        FileStatus status = this.getNullableFileStatus(currentPath);
        if (status != null) {
            if (!status.isDirectory()) {
                throw new FileAlreadyExistsException("Cannot mkdir, file at path already exists: " + path);
            }
            LOG.debug("Path already exists, nothing to create");
            return true;
        }
        ArrayList<Path> directoriesToCreate = new ArrayList<Path>();
        while (status == null) {
            directoriesToCreate.add(currentPath);
            currentPath = currentPath.getParent();
            status = this.getNullableFileStatus(currentPath);
        }
        if (!status.isDirectory()) {
            throw new ParentNotDirectoryException("Found a parent path that is not a directory: " + status.getPath());
        }
        LOG.debug("Attempting to create directories: {}", directoriesToCreate);
        for (Path directoryToCreate : directoriesToCreate) {
            this.dataStore.createDirectory(directoryToCreate);
        }
        return true;
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        LOG.debug("Opening path {}, bufferSize {}", (Object)path, (Object)bufferSize);
        FileStatus status = this.getFileStatus(path);
        if (status.isDirectory()) {
            throw new FileNotFoundException("File at path location is a directory: " + path);
        }
        return new FSDataInputStream((InputStream)this.dataStore.openReadStream(status, path, bufferSize, this.statistics));
    }

    public boolean rename(Path source, Path destination) throws IOException {
        Path destinationPathToUse;
        FileStatus sourceStatus;
        LOG.debug("Renaming {} to {}", (Object)source, (Object)destination);
        Path absoluteSource = this.ensureAbsolutePath(source);
        Path absoluteDestination = this.ensureAbsolutePath(destination);
        try {
            sourceStatus = this.getFileStatus(absoluteSource);
        }
        catch (FileNotFoundException e) {
            LOG.debug("Source file not found");
            return false;
        }
        if (sourceStatus.getPath().isRoot()) {
            LOG.debug("Cannot rename root");
            return false;
        }
        if (absoluteSource.equals((Object)absoluteDestination)) {
            if (sourceStatus.isDirectory()) {
                LOG.debug("Destination is the same as source, renaming directory to itself not allowed");
                return false;
            }
            LOG.debug("Destination is the same as source, renaming file to itself is allowed");
            return true;
        }
        FileStatus destinationStatus = this.getNullableFileStatus(absoluteDestination);
        if (destinationStatus == null) {
            FileStatus destinationParentStatus = this.getNullableFileStatus(absoluteDestination.getParent());
            if (destinationParentStatus == null || destinationParentStatus.isFile()) {
                LOG.debug("Destination parent directory does not exist, or is a file");
                return false;
            }
            destinationPathToUse = absoluteDestination;
        } else {
            if (destinationStatus.isFile()) {
                LOG.debug("Destination exists and is a file");
                return false;
            }
            destinationPathToUse = new Path(absoluteDestination, absoluteSource.getName());
        }
        if (absoluteSource.equals((Object)destinationPathToUse)) {
            if (sourceStatus.isDirectory()) {
                LOG.debug("Resolved destination is the same as source, renaming directory to itself not allowed");
                return false;
            }
            LOG.debug("Resolved destination is the same as source, renaming file to itself is allowed");
            return true;
        }
        if (this.isDescendant(absoluteSource, absoluteDestination)) {
            LOG.debug("Destination cannot be a child of src");
            return false;
        }
        if (sourceStatus.isFile()) {
            LOG.debug("Renaming file {} to {}", (Object)absoluteSource, (Object)destinationPathToUse);
            this.dataStore.renameFile(absoluteSource, destinationPathToUse);
        } else {
            LOG.debug("Renaming directory {} to {}", (Object)absoluteSource, (Object)destinationPathToUse);
            this.dataStore.renameDirectory(absoluteSource, destinationPathToUse);
        }
        return true;
    }

    private boolean isDescendant(Path source, Path destination) {
        String destinationPath;
        String sourcePath = source.toUri().getPath();
        if (!sourcePath.endsWith("/")) {
            sourcePath = sourcePath + "/";
        }
        return sourcePath.equals(destinationPath = destination.toUri().getPath()) || destinationPath.startsWith(sourcePath);
    }

    public long getDefaultBlockSize() {
        return this.dataStore.getBlockSizeInBytes();
    }

    public int getDefaultPort() {
        return -1;
    }

    public String getCanonicalServiceName() {
        return null;
    }

    private Path ensureAbsolutePath(Path path) {
        if (path.isAbsolute()) {
            return path;
        }
        return new Path(this.workingDirectory, path);
    }

    public Path getWorkingDirectory() {
        return this.workingDirectory;
    }

    public void setWorkingDirectory(Path workingDirectory) {
        this.workingDirectory = workingDirectory;
    }

    BmcDataStore getDataStore() {
        return this.dataStore;
    }

    public URI getUri() {
        return this.uri;
    }

    private static final class PathLengthComparator
    implements Comparator<Path> {
        private PathLengthComparator() {
        }

        @Override
        public int compare(Path path1, Path path2) {
            return Integer.compare(path2.toUri().toString().length(), path1.toUri().toString().length());
        }
    }

    @VisibleForTesting
    static class UriParser {
        private static final Pattern URI_PATTERN = Pattern.compile("^(?:oci|oraclebmc):\\/\\/([^:\\/]+)@([^:\\/]+)");
        private final URI uri;
        private final Matcher uriMatcher;

        UriParser(URI uri) {
            this.uri = uri;
            this.uriMatcher = URI_PATTERN.matcher(uri.toString());
            if (!this.uriMatcher.find() || this.uriMatcher.groupCount() != 2) {
                throw new IllegalArgumentException("Unknown uri pattern: " + uri.toString());
            }
        }

        String getScheme() {
            return this.uri.getScheme();
        }

        String getAuthority() {
            return this.uri.getAuthority();
        }

        String getNamespace() {
            String namespace = this.uri.getHost();
            if (namespace != null) {
                return namespace.trim();
            }
            return this.uriMatcher.group(2).trim();
        }

        String getBucket() {
            String bucket = this.uri.getUserInfo();
            if (bucket != null) {
                return bucket.trim();
            }
            return this.uriMatcher.group(1).trim();
        }
    }
}

