/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.client;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.scm.net.InnerNode;
import org.apache.hadoop.hdds.scm.net.NetworkTopology;
import org.apache.hadoop.hdds.scm.net.NetworkTopologyImpl;
import org.apache.hadoop.hdds.scm.protocol.ScmBlockLocationProtocol;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScmTopologyClient {
    private static final Logger LOG = LoggerFactory.getLogger(ScmTopologyClient.class);
    private final ScmBlockLocationProtocol scmBlockLocationProtocol;
    private final AtomicReference<NetworkTopology> cache = new AtomicReference();
    private ScheduledExecutorService executorService;

    public ScmTopologyClient(ScmBlockLocationProtocol scmBlockLocationProtocol) {
        this.scmBlockLocationProtocol = scmBlockLocationProtocol;
    }

    public NetworkTopology getClusterMap() {
        return Objects.requireNonNull(this.cache.get(), "ScmBlockLocationClient must have been initialized already.");
    }

    public void start(ConfigurationSource conf) throws IOException {
        InnerNode initialTopology = this.scmBlockLocationProtocol.getNetworkTopology();
        LOG.info("Initial network topology fetched from SCM: {}.", (Object)initialTopology);
        this.cache.set((NetworkTopology)new NetworkTopologyImpl(conf.get("ozone.scm.network.topology.schema.file", "network-topology-default.xml"), initialTopology));
        this.scheduleNetworkTopologyPoller(conf, Instant.now());
    }

    public void stop() {
        if (this.executorService != null) {
            this.executorService.shutdown();
            try {
                if (this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                LOG.error("Interrupted while shutting down executor service.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        }
    }

    private void scheduleNetworkTopologyPoller(ConfigurationSource conf, Instant initialInvocation) {
        Duration refreshDuration = ScmTopologyClient.parseRefreshDuration(conf);
        Instant nextRefresh = initialInvocation.plus(refreshDuration);
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("NetworkTopologyPoller").setDaemon(true).build();
        this.executorService = Executors.newScheduledThreadPool(1, threadFactory);
        Duration initialDelay = Duration.between(Instant.now(), nextRefresh);
        LOG.debug("Scheduling NetworkTopologyPoller with an initial delay of {}.", (Object)initialDelay);
        this.executorService.scheduleAtFixedRate(() -> this.checkAndRefresh(conf), initialDelay.toMillis(), refreshDuration.toMillis(), TimeUnit.MILLISECONDS);
    }

    public static Duration parseRefreshDuration(ConfigurationSource conf) {
        long refreshDurationInMs = conf.getTimeDuration("ozone.om.network.topology.refresh.duration", "1h", TimeUnit.MILLISECONDS);
        return Duration.ofMillis(refreshDurationInMs);
    }

    private synchronized void checkAndRefresh(ConfigurationSource conf) {
        InnerNode current = (InnerNode)this.cache.get().getNode("");
        try {
            InnerNode newTopology = this.scmBlockLocationProtocol.getNetworkTopology();
            if (!newTopology.equals((Object)current)) {
                this.cache.set((NetworkTopology)new NetworkTopologyImpl(conf.get("ozone.scm.network.topology.schema.file", "network-topology-default.xml"), newTopology));
                LOG.info("Updated network topology fetched from SCM: {}.", (Object)newTopology);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Error fetching updated network topology from SCM", e);
        }
    }
}

