"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.boundary = void 0;
const x6_common_1 = require("@antv/x6-common");
const x6_geometry_1 = require("@antv/x6-geometry");
const util_1 = require("./util");
const util_2 = require("../../util");
/**
 * Places the connection point at the intersection between the
 * edge path end segment and the actual shape of the target magnet.
 */
const boundary = function (line, view, magnet, options) {
    let node;
    let intersection;
    const anchor = line.end;
    const selector = options.selector;
    if (typeof selector === 'string') {
        node = view.findOne(selector);
    }
    else if (Array.isArray(selector)) {
        node = x6_common_1.ObjectExt.getByPath(magnet, selector);
    }
    else {
        node = (0, util_1.findShapeNode)(magnet);
    }
    if (!x6_common_1.Dom.isSVGGraphicsElement(node)) {
        if (node === magnet || !x6_common_1.Dom.isSVGGraphicsElement(magnet)) {
            return anchor;
        }
        node = magnet;
    }
    const localShape = view.getShapeOfElement(node);
    const magnetMatrix = view.getMatrixOfElement(node);
    const translateMatrix = view.getRootTranslatedMatrix();
    const rotateMatrix = view.getRootRotatedMatrix();
    const targetMatrix = translateMatrix
        .multiply(rotateMatrix)
        .multiply(magnetMatrix);
    const localMatrix = targetMatrix.inverse();
    const localLine = util_2.Util.transformLine(line, localMatrix);
    const localRef = localLine.start.clone();
    const data = view.getDataOfElement(node);
    if (options.insideout === false) {
        if (data.shapeBBox == null) {
            data.shapeBBox = localShape.bbox();
        }
        const localBBox = data.shapeBBox;
        if (localBBox != null && localBBox.containsPoint(localRef)) {
            return anchor;
        }
    }
    if (options.extrapolate === true) {
        localLine.setLength(1e6);
    }
    // Caching segment subdivisions for paths
    let pathOptions;
    if (x6_geometry_1.Path.isPath(localShape)) {
        const precision = options.precision || 2;
        if (data.segmentSubdivisions == null) {
            data.segmentSubdivisions = localShape.getSegmentSubdivisions({
                precision,
            });
        }
        pathOptions = {
            precision,
            segmentSubdivisions: data.segmentSubdivisions,
        };
        intersection = localLine.intersect(localShape, pathOptions);
    }
    else {
        intersection = localLine.intersect(localShape);
    }
    if (intersection) {
        if (Array.isArray(intersection)) {
            intersection = localRef.closest(intersection);
        }
    }
    else if (options.sticky === true) {
        // No intersection, find the closest point instead
        if (x6_geometry_1.Rectangle.isRectangle(localShape)) {
            intersection = localShape.getNearestPointToPoint(localRef);
        }
        else if (x6_geometry_1.Ellipse.isEllipse(localShape)) {
            intersection = localShape.intersectsWithLineFromCenterToPoint(localRef);
        }
        else {
            intersection = localShape.closestPoint(localRef, pathOptions);
        }
    }
    const cp = intersection
        ? util_2.Util.transformPoint(intersection, targetMatrix)
        : anchor;
    let cpOffset = options.offset || 0;
    if (options.stroked !== false) {
        if (typeof cpOffset === 'object') {
            cpOffset = Object.assign({}, cpOffset);
            if (cpOffset.x == null) {
                cpOffset.x = 0;
            }
            cpOffset.x += (0, util_1.getStrokeWidth)(node) / 2;
        }
        else {
            cpOffset += (0, util_1.getStrokeWidth)(node) / 2;
        }
    }
    return (0, util_1.offset)(cp, line.start, cpOffset);
};
exports.boundary = boundary;
//# sourceMappingURL=boundary.js.map