/*
 * IncludeTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.4
 * Created by suhler on 99/05/06
 * Last modified by suhler on 04/08/30 10:29:38
 */

package sunlabs.brazil.template;

import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Format;
import sunlabs.brazil.util.http.HttpInputStream;
import sunlabs.brazil.util.http.HttpRequest;
import sunlabs.brazil.util.http.HttpRequest;
import sunlabs.brazil.util.http.MimeHeaders;
import sunlabs.brazil.util.http.HttpUtil;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import java.util.Enumeration;
import java.util.StringTokenizer;

/**
 * Template class for substituting html pages into an html page.
 * This class is used by the TemplateHandler.  This does not perform
 * traditional server-side include processing, whose normal purpose is to
 * include standard headers and footers.  That functionallity is provided
 * instead by including the content into the template, and not the other
 * way around.
 * <p>
 * This include incorporates entire pages from other sites.
 * <dl class=props>
 * <dt>proxy	<dd>If specified, connect through a proxy. This should be
 *		in the form <code>host:port</code>, or <code>host</code>
 *		it the desired port is <code>80</code>.
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		@(#)IncludeTemplate.java	2.4
 */
    /**
     * Convert the html tag "include" in to text for an included
     * html page.
     * Attributes processed
     * <dl class=attributes>
     * <dt>href<dd>Absolute url to fetch, and insert here
     * <dt>post<dd>Post data if any.  If set, an http POST is issued.
     * The post data is expected to be in www-url-encoded format.
     * <dt>alt<dd>Text to insert if URL can't be obtained.
     * <dt>name<dd>The name of the variable to put the result in.
     * If this is specified, the content is not included in place.
     * The template prefix is automatically prepended to the name.
     * <dt>proxy<dd>The proxy:port to use as a proxy (if any).
     * If specified, it overrides the <code>proxy</code> property, 
     * in <code>request.props</code>.
     * <dt>addheaders<dd>A white space delimited set of token names that
     * represent additional http headers to add to the target request.
     * For each token, the values <i>[token].name</i> and
     * <i>[token].value</i> in the
     * <code>request.props</code> are used for the header name and value
     * respectively.
     * <dt>encoding<dd>The character encoding to use if it can't be
     * automatically determined.   Defaults to the default platform encoding.
     * <dt>getheaders<dd>The name of the variable prefix to use to
     * extract the http response headers.
     * If this not specified, no response headers are retrieved.
     * The result will be properties of the form:
     * <code>[prefix].[getheaders].[header_name]=[header_value]</code>.
     * If multiple entries exist for a particular header name, the values
     * are combined as per HTTP conventions (e.g. v1, v2, ... vn).
     * The pseudo headers <code>status</code> and <code>encoding</code>
     * will contain the http status line and charset encoding, respectively.
     * <dt>error<dd>If <i>name</i> is specified and the result
     * could not be obtained, the error message is placed here.
     * <dt>queryList<dd>A list of property names whose names and values
     * will be added to the url as query parameters.  The values are properly
     * url-encoded.
     * </dl>
     * Example: <code>&lt;include href=http://www.foo.com/index.html&gt;</code>
     */

public class IncludeTemplate
    extends Template {

    public void
    tag_include(RewriteContext hr) {
	String href = hr.get("href");	// the url to fetch
	String alt = hr.get("alt");	// the result if fetch failed
	String name = hr.get("name");	// the variable to put the result in
	String post = hr.get("post");	// post data (if any)
	String addheaders = hr.get("addheaders"); // additional http headers
	String getheaders = hr.get("getheaders"); // retrieve response headers
	String encoding = hr.get("encoding", null); // retrieve response headers
	boolean postInline = hr.isTrue("postinline");  // inline post data
	// these are experimental
	boolean encodeQuery = hr.isTrue("encodeQuery");  // fix the url
	String queryList = hr.get("queryList");

	if (alt==null) {
	    alt = "";
	}
	String proxy = hr.get("proxy", "");
	String proxyHost = null;
	int proxyPort = 80;
	if (!proxy.equals("")) {
	    int index = proxy.indexOf(":");
	    if (index < 0) {
		proxyHost = proxy;
	    } else {
		proxyHost = proxy.substring(0,index);
		try {
		    proxyPort =
			Integer.decode(proxy.substring(index+1)).intValue();
		} catch (Exception e) {}
	    }
	}

	debug(hr);
	hr.killToken();

	// XXX This is wrong!

	if (postInline && !hr.isSingleton()) {
            StringBuffer sb = new StringBuffer();
	    int nest=0;
            while (hr.nextToken()) {
                String token = hr.getToken();
		hr.killToken();
                hr.getTag();   // force singleton checking!
                if (token.startsWith("<include ") && !hr.isSingleton()) {
                    nest++;
                } else if (token.equals("</include>") && (--nest < 0)) {
                    break;
                }
                sb.append(token);
            }
	    // hr.append(Format.subst(hr.request.props, hr.getBody(), true));
	    post = Format.subst(hr.request.props, sb.toString());
	    hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		"Posting: " + post);
	    debug(hr, post);
	}

	MimeHeaders clientHeaders = new MimeHeaders();
	hr.request.headers.copyTo(clientHeaders);
	HttpRequest.removePointToPointHeaders(clientHeaders, false);
	clientHeaders.remove("host");

	/*
	 * Fix broken query strings - this is dubious
	 */

	if (encodeQuery) {
	    int index = href.indexOf("?");
	    if (index > 0) {
		href = href.substring(0,index+1) + 
			href.substring(index+1).replace(' ', '+');
	    }
	}

	/*
	 * build a query string based on properties
	 */

	if (queryList != null) {
	    StringTokenizer st = new StringTokenizer(queryList);
            StringBuffer sb = new StringBuffer(href);
	    String delim = href.indexOf("?") > 0 ? "&" : "?";
	    while (st.hasMoreTokens()) {
	        String key = st.nextToken();
	        String value = hr.request.props.getProperty(key);
		sb.append(delim).append(key).append("=");
		if (value != null) {
		    sb.append(HttpUtil.urlEncode(value));
		}
		delim="&";
	    }
	    href = sb.toString();
	    debug(hr, "Using href: " + href);
	}

	HttpRequest target = new HttpRequest(href);
	clientHeaders.copyTo(target.requestHeaders);
	if (addheaders != null) {
	    target.addHeaders(addheaders, hr.request.props);
	}
	target.requestHeaders.putIfNotPresent("Host",
	    HttpUtil.extractUrlHost(href));
	target.setRequestHeader("Via", "Brazil-Include/2.0");

	if (proxyHost != null) {
	    debug(hr, "Using proxy: " + proxy);
	    target.setProxy(proxyHost, proxyPort);
	}
	hr.request.log(Server.LOG_INFORMATIONAL, hr.prefix,
		"Fetching: " + href);
	try {
	    if (post != null) {
		OutputStream out = target.getOutputStream();
		out.write(post.getBytes());
		out.close();
	    }
	    String enc = target.getContent(encoding);
	    if (name != null) {
		hr.request.props.put(hr.prefix + name, enc);
	    } else {
		hr.append(enc);
	    }
	    if (getheaders != null) {
	        Enumeration keys = target.responseHeaders.keys();
	        while(keys.hasMoreElements()) {
		    String key = (String) keys.nextElement();
		    String full = hr.prefix + getheaders + "." + key;
		    if (hr.request.props.containsKey(full)) {
		        hr.request.props.put(full,
				hr.request.props.getProperty(full) +
			        ", " + target.responseHeaders.get(key));
		    } else {
		        hr.request.props.put(full,
			        target.responseHeaders.get(key));
		    }
	        }
		hr.request.props.put(hr.prefix + getheaders + ".status",
			target.status);
		String en = target.getEncoding();
		if (en != null) {
		    hr.request.props.put(hr.prefix + getheaders +
			    ".encoding", en);
		}
	    }
	} catch (IOException e) {
	    if (name != null) {
		hr.request.props.put(hr.prefix + name, alt);
		hr.request.props.put(hr.prefix + "error", e.getMessage());
	    } else {
	        hr.append("<!-- " + e + " -->" + alt);
	    }
	}
	hr.request.log(Server.LOG_INFORMATIONAL, hr.prefix,
		"Done fetching: " + href + " " + target.status);
    }

    /**
     * Treat comments of the form:
     * <code>&lt;!#include ...&gt;</code> as if they were include tags.
     */

    public void
    comment(RewriteContext hr)
    {
	if (hr.getBody().startsWith("#include") == false) {
	    return;
	}
	tag_include(hr);
    }
}
