/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook.repo;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientURI;
import com.mongodb.client.AggregateIterable;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.InsertManyOptions;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.Updates;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.NoteParser;
import org.apache.zeppelin.notebook.repo.AbstractNotebookRepo;
import org.apache.zeppelin.notebook.repo.AutoLock;
import org.apache.zeppelin.notebook.repo.AutoReadWriteLock;
import org.apache.zeppelin.notebook.repo.NotebookRepoSettingsInfo;
import org.apache.zeppelin.notebook.repo.VFSNotebookRepo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MongoNotebookRepo
extends AbstractNotebookRepo {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoNotebookRepo.class);
    private MongoClient client;
    private MongoDatabase db;
    private MongoCollection<Document> notes;
    private MongoCollection<Document> folders;
    private String folderName;
    private AutoReadWriteLock lock = new AutoReadWriteLock();

    public void init(ZeppelinConfiguration zConf, NoteParser noteParser) throws IOException {
        super.init(zConf, noteParser);
        this.client = new MongoClient(new MongoClientURI(zConf.getMongoUri()));
        this.db = this.client.getDatabase(zConf.getMongoDatabase());
        this.notes = this.db.getCollection(zConf.getMongoCollection());
        this.folderName = zConf.getMongoFolder();
        this.folders = this.db.getCollection(this.folderName);
        if (zConf.getMongoAutoimport()) {
            this.insertFileSystemNotes();
        }
    }

    private void insertFileSystemNotes() throws IOException {
        try (VFSNotebookRepo vfsRepo = new VFSNotebookRepo();){
            vfsRepo.init(this.zConf, this.noteParser);
            Map infos = vfsRepo.list(null);
            try (AutoLock autoLock = this.lock.lockForWrite();){
                for (NoteInfo info : infos.values()) {
                    Note note = vfsRepo.get(info.getId(), info.getPath(), null);
                    this.saveOrIgnore(note, null);
                }
            }
        }
    }

    public Map<String, NoteInfo> list(AuthenticationInfo subject) throws IOException {
        LOGGER.debug("list repo.");
        HashMap<String, NoteInfo> infos = new HashMap<String, NoteInfo>();
        Document match = new Document("$match", (Object)new Document("isDir", (Object)false));
        Document graphLookup = new Document("$graphLookup", (Object)new Document("from", (Object)this.folderName).append("startWith", (Object)"$pId").append("connectFromField", (Object)"pId").append("connectToField", (Object)"_id").append("as", (Object)"fullPath"));
        try (AutoLock autoLock = this.lock.lockForRead();){
            List<Document> list = Arrays.asList(match, graphLookup);
            AggregateIterable aggregate = this.folders.aggregate(list);
            for (Document document : aggregate) {
                String id = document.getString((Object)"_id");
                String name = document.getString((Object)"name");
                List fullPath = (List)document.get((Object)"fullPath", List.class);
                fullPath.sort(Comparator.comparing(pathNode -> pathNode.getString((Object)"_id")));
                StringBuilder sb = new StringBuilder();
                for (Document pathNode2 : fullPath) {
                    sb.append("/").append(pathNode2.getString((Object)"name"));
                }
                String fullPathStr = sb.append("/").append(name).toString();
                NoteInfo noteInfo = new NoteInfo(id, fullPathStr);
                infos.put(id, noteInfo);
            }
        }
        return infos;
    }

    public Note get(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("get note, noteId: {}, notePath:{}", (Object)noteId, (Object)notePath);
        return this.getNote(noteId, notePath);
    }

    private Note getNote(String noteId, String notePath) throws IOException {
        Document doc = (Document)this.notes.find(Filters.eq((String)"_id", (Object)noteId)).first();
        if (doc == null) {
            throw new IOException("Note '" + noteId + "' in path '" + notePath + "'not found");
        }
        return this.documentToNote(noteId, doc);
    }

    public void save(Note note, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("save note, note: {}", (Object)note);
        String[] pathArray = this.toPathArray(note.getPath(), false);
        try (AutoLock autoLock = this.lock.lockForWrite();){
            String pId = this.completeFolder(pathArray);
            this.saveNote(note);
            this.saveNotePath(note.getId(), note.getName(), pId);
        }
    }

    private void saveOrIgnore(Note note, AuthenticationInfo subject) {
        try {
            String[] pathArray = this.toPathArray(note.getPath(), false);
            String pId = this.completeFolder(pathArray);
            this.saveNoteOrIgnore(note);
            this.saveNotePathOrIgnore(note.getId(), note.getName(), pId);
        }
        catch (Exception e) {
            LOGGER.warn("ignore error when insert note '{}': {}", (Object)note, (Object)e.getMessage());
        }
    }

    private void saveNotePath(String noteId, String noteName, String pId) {
        Document filter = new Document("_id", (Object)noteId);
        Document doc = new Document("_id", (Object)noteId).append("pId", (Object)pId).append("isDir", (Object)false).append("name", (Object)noteName);
        this.folders.replaceOne((Bson)filter, (Object)doc, new UpdateOptions().upsert(true));
    }

    private void saveNotePathOrIgnore(String noteId, String noteName, String pId) {
        Document doc = new Document("_id", (Object)noteId).append("pId", (Object)pId).append("isDir", (Object)false).append("name", (Object)noteName);
        this.folders.insertMany(Collections.singletonList(doc), new InsertManyOptions().ordered(false));
    }

    private void saveNote(Note note) {
        Document doc = this.noteToDocument(note);
        this.notes.replaceOne(Filters.eq((String)"_id", (Object)note.getId()), (Object)doc, new UpdateOptions().upsert(true));
    }

    private void saveNoteOrIgnore(Note note) {
        Document doc = this.noteToDocument(note);
        this.notes.insertMany(Collections.singletonList(doc), new InsertManyOptions().ordered(false));
    }

    public void move(String noteId, String notePath, String newNotePath, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("move note, noteId: {}, notePath: {}, newNotePath: {}", new Object[]{noteId, notePath, newNotePath});
        if (StringUtils.equals((CharSequence)notePath, (CharSequence)newNotePath)) {
            return;
        }
        String[] pathArray = this.toPathArray(newNotePath, true);
        String[] parentPathArray = Arrays.copyOfRange(pathArray, 0, pathArray.length - 1);
        String noteName = pathArray[pathArray.length - 1];
        try (AutoLock autoLock = this.lock.lockForWrite();){
            String pId = this.completeFolder(parentPathArray);
            this.moveNote(noteId, pId, noteName);
        }
    }

    private void moveNote(String noteId, String parentId, String noteName) {
        Document doc = new Document("$set", (Object)new Document("pId", (Object)parentId).append("name", (Object)noteName));
        this.folders.updateOne(Filters.eq((String)"_id", (Object)noteId), (Bson)doc);
        this.notes.updateOne(Filters.eq((String)"_id", (Object)noteId), Updates.set((String)"name", (Object)noteName));
    }

    public void move(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("move folder, folderPath: {}, newFolderPath: {}", (Object)folderPath, (Object)newFolderPath);
        if (StringUtils.equals((CharSequence)folderPath, (CharSequence)newFolderPath)) {
            return;
        }
        String[] pathArray = this.toPathArray(folderPath, true);
        String[] newPathArray = this.toPathArray(newFolderPath, true);
        String[] newFolderParentArray = Arrays.copyOfRange(newPathArray, 0, newPathArray.length - 1);
        try (AutoLock autoLock = this.lock.lockForWrite();){
            String id = this.findFolder(pathArray);
            String newPId = this.completeFolder(newFolderParentArray);
            String newFolderName = newPathArray[newPathArray.length - 1];
            Document doc = new Document("$set", (Object)new Document("_id", (Object)id).append("pId", (Object)newPId).append("isDir", (Object)true).append("name", (Object)newFolderName));
            this.folders.updateOne(Filters.eq((String)"_id", (Object)id), (Bson)doc);
        }
    }

    public void remove(String noteId, String notePath, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("remove note, noteId:{}, notePath:{}", (Object)noteId, (Object)notePath);
        try (AutoLock autoLock = this.lock.lockForWrite();){
            this.folders.deleteOne(Filters.eq((String)"_id", (Object)noteId));
            this.notes.deleteOne(Filters.eq((String)"_id", (Object)noteId));
            String[] pathArray = this.toPathArray(notePath, false);
            for (int i = pathArray.length; i >= 0; --i) {
                boolean isEmpty;
                String[] current = Arrays.copyOfRange(pathArray, 0, i);
                String folderId = this.findFolder(current);
                boolean bl = isEmpty = this.folders.count(Filters.eq((String)"pId", (Object)folderId)) <= 0L;
                if (isEmpty) {
                    this.folders.deleteOne(Filters.eq((String)"_id", (Object)folderId));
                    continue;
                }
                break;
            }
        }
    }

    public void remove(String folderPath, AuthenticationInfo subject) throws IOException {
        LOGGER.debug("remove folder, folderPath: {}", (Object)folderPath);
        String[] pathArray = this.toPathArray(folderPath, true);
        try (AutoLock autoLock = this.lock.lockForWrite();){
            String id = this.findFolder(pathArray);
            FindIterable iter = this.folders.find(Filters.eq((String)"pId", (Object)id));
            for (Document node : iter) {
                String nodeId = node.getString((Object)"_id");
                Boolean isDir = node.getBoolean((Object)"isDir");
                String nodeName = node.getString((Object)"name");
                if (isDir.booleanValue()) {
                    StringBuilder sb = new StringBuilder();
                    for (String s : pathArray) {
                        sb.append("/").append(s);
                    }
                    sb.append("/").append(nodeName);
                    String nodePath = sb.toString();
                    this.remove(nodePath, subject);
                } else {
                    this.folders.deleteOne(Filters.eq((String)"_id", (Object)nodeId));
                    this.notes.deleteOne(Filters.eq((String)"_id", (Object)nodeId));
                }
                this.folders.deleteOne(Filters.eq((String)"_id", (Object)nodeId));
            }
        }
    }

    public void close() {
        this.client.close();
    }

    public List<NotebookRepoSettingsInfo> getSettings(AuthenticationInfo subject) {
        LOGGER.warn("Method not implemented");
        return Collections.emptyList();
    }

    public void updateSettings(Map<String, String> settings, AuthenticationInfo subject) {
        LOGGER.warn("Method not implemented");
    }

    private String completeFolder(String[] splitPath) {
        String pId = "0";
        for (String currentPath : splitPath) {
            Document query = new Document("pId", (Object)pId).append("isDir", (Object)true).append("name", (Object)currentPath);
            String cId = new ObjectId().toString();
            Document doc = new Document("$setOnInsert", (Object)new Document("_id", (Object)cId).append("pId", (Object)pId).append("isDir", (Object)true).append("name", (Object)currentPath));
            Document exist = (Document)this.folders.find((Bson)query).first();
            if (exist == null) {
                this.folders.updateOne((Bson)query, (Bson)doc, new UpdateOptions().upsert(true));
                pId = cId;
                continue;
            }
            pId = exist.getString((Object)"_id");
        }
        return pId;
    }

    private String findFolder(String[] splitPath) {
        String pId = "0";
        if (splitPath.length == 1 && "".equals(splitPath[0]) || ArrayUtils.isEmpty((Object[])splitPath)) {
            return pId;
        }
        for (String currentPath : splitPath) {
            Bson where = Filters.and((Bson[])new Bson[]{Filters.eq((String)"pId", (Object)pId), Filters.eq((String)"isDir", (Object)true), Filters.eq((String)"name", (Object)currentPath)});
            Document node = (Document)this.folders.find(where).first();
            if (null == node) {
                throw new IllegalStateException("folder not found in path:" + currentPath);
            }
            pId = node.getString((Object)"_id");
        }
        return pId;
    }

    String[] toPathArray(String notePath, boolean includeLast) {
        if (null == notePath || notePath.length() == 0) {
            throw new NullPointerException("notePath is null");
        }
        if ("/".equals(notePath = notePath.replaceAll("/+", "/"))) {
            return ArrayUtils.EMPTY_STRING_ARRAY;
        }
        if (notePath.startsWith("/")) {
            notePath = notePath.substring(1);
        }
        String[] arr = notePath.split("/");
        return includeLast ? arr : Arrays.copyOfRange(arr, 0, arr.length - 1);
    }

    private Note documentToNote(String noteId, Document doc) throws IOException {
        String json = doc.toJson();
        return this.noteParser.fromJson(noteId, json);
    }

    private Document noteToDocument(Note note) {
        String json = note.toJson();
        Document doc = Document.parse((String)json);
        doc.put("_id", (Object)note.getId());
        return doc;
    }

    private class Fields {
        private static final String ID = "_id";
        private static final String NAME = "name";
        private static final String IS_DIR = "isDir";
        private static final String PID = "pId";
        private static final String FULL_PATH = "fullPath";

        private Fields() {
        }
    }
}

